/*
 *
 *  Copyright (C) 2008-2009 RICOH Co.,LTD.
 *  All rights reserved.
 *
 *  affiliation	:DSS Development Center
 *  		 Document Solutions & Services Division
 * 
 *  purpose	: SCF waitForCard sample.
 *
 */

package scfwait;

import jp.co.ricoh.dsdk.panel.AttributeType;
import jp.co.ricoh.dsdk.panel.Container;
import jp.co.ricoh.dsdk.panel.Font;
import jp.co.ricoh.dsdk.panel.Frame;
import jp.co.ricoh.dsdk.panel.LED;
import jp.co.ricoh.dsdk.panel.LEDExecutionException;
import jp.co.ricoh.dsdk.panel.Label;
import jp.co.ricoh.dsdk.panel.Text;
import jp.co.ricoh.dsdk.panel.Window;
import jp.co.ricoh.dsdk.panel.LED.Operation;
import jp.co.ricoh.dsdk.panel.LED.Type;
import jp.co.ricoh.dsdk.panel.event.KeyAdapter;
import jp.co.ricoh.dsdk.panel.event.KeyEvent;
import jp.co.ricoh.dsdk.scard.framework.card.Card;
import jp.co.ricoh.dsdk.scard.framework.card.CardAccessException;
import jp.co.ricoh.dsdk.scard.framework.card.CardManager;
import jp.co.ricoh.dsdk.scard.framework.card.InitializeException;
import jp.co.ricoh.dsdk.scard.framework.card.TimeoutException;
import jp.co.ricoh.dsdk.scard.framework.service.CardServiceException;
import jp.co.ricoh.dsdk.scard.framework.service.CardServiceNotFoundException;
import jp.co.ricoh.dsdk.scard.option.file.FileAccessService;
import jp.co.ricoh.dsdk.scard.option.file.FilePath;
import jp.co.ricoh.dsdk.scard.option.security.VerificationException;
import jp.co.ricoh.dsdk.xlet.UnavailableContainerException;
import jp.co.ricoh.dsdk.xlet.Xlet;
import jp.co.ricoh.dsdk.xlet.XletContext;
import jp.co.ricoh.dsdk.xlet.XletStateChangeException;

/**
 * ScfWait
 * This is a sample application using the waitForCard method of the SmartCard Framework.
 * 
 * ScfWait
 * SmartCard Frameworkで、waitForCardメソッドを利用するサンプルプログラムです。
 */
public class ScfWait implements Xlet {
	
	/**
	 * Indicates the file path that stores the serial number of the card.
	 * 
	 * シリアル番号が格納されたファイル
	 */
	private static final FilePath filePath = new FilePath(":3F00:0002", ':');
	
	/**
	 * Indicates the length of the file that stores the serial number of the card.
	 * 
	 * シリアル番号が格納されたファイルの長さ
	 */
	private static final int fileLen = 8;
	
	/**
	 * The CardManager object.
	 * 
	 * CardManagerオブジェクト 
	 */
	private CardManager cardManager;
	
	/**
	 * The thread to be used to wait for cards.
	 * 
	 * waitForCardを実行するスレッド
	 */
	private Thread thread;

	/**
	 * The root frame.
	 * 
	 * ルートフレーム
	 */
	private Frame frame;
	
	/**
	 * The label to be used to display a message.
	 * 
	 * メッセージ表示ラベル
	 */
	private Label msgLabel;
	
	/**
	 * The label to be used to display a serial number.
	 * 
	 * シリアル番号表示ラベル
	 */
	private Label serialLabel;
	
	/**
	 * This is an initialization process.
	 * 1.obtains the root frame.
	 * 2.creates GUI.
	 * 3.creates a CardManager object.
	 * 4.sets LED state.
	 *  
	 * 初期化処理です。
	 * 1.ルートフレームを取得する。
	 * 2.GUIを作成する。
	 * 3.CardManagerオブジェクトを生成する。
	 * 4.LEDを設定する。
	 */
	synchronized public void initXlet(XletContext xletContext) throws XletStateChangeException {	
		try {
			Container parent = xletContext.getContainer();
			while (!(parent instanceof Frame)) {
				parent = parent.getParent();
				if (parent == null) break;
			}
			if (parent == null) {
				return;
			}
			frame = (Frame) parent;

			createGUI();
			
		} catch (UnavailableContainerException e) {
            throw new XletStateChangeException(e.getMessage());

        } catch (Exception e){
            throw new XletStateChangeException(e.getMessage());
        }
        
        try {
			cardManager = new CardManager();
						
		} catch (InitializeException e) {
			msgLabel.setText(new Text("ERR_INIT"));
			e.printStackTrace();
		}
		
		setLED(Type.START, Operation.GRN_ON);
	}

	/**
	 * Activation
	 * 
	 * 活性化。
	 */
	synchronized public void startXlet() throws XletStateChangeException {
	}
	
	/**
	 * Stop.
	 * 
	 * 停止。
	 */
	synchronized public void pauseXlet() {
	}

	/**
	 * End process.
	 * This process is similar to the process followed after the clear/stop key is pressed.
	 * 
	 * 終了処理。
	 * クリアストップキー押された時と同様の処理を行う。
	 * 
	 */
	synchronized public void destroyXlet(boolean destroy) throws XletStateChangeException {
		pushClearStopKey();
	}
	
	/**
	 * GUI creation.
	 * This is called from the initXlet method, and creates GUI on the operation panel.
	 * 1.creates a title window.
	 * 2.creates a title label.
	 * 3.creates a message label.
	 * 4.creates a serial number label.
	 * 5.registers a KeyListener to the root frame for processing KeyEvents. 
	 * 
	 * GUI作成。
	 * initXletメソッドから呼び出され、オペパネにＧＵＩを作成します。
	 * 1.タイトル表示ウィンドウを作成する。
	 * 2.タイトル表示ラベルを作成する。
	 * 3.メッセージ表示ラベルを作成する。
	 * 4.シリアル番号表示ラベルを作成する。
	 * 5.キーリスナーを登録して、キーイベントを処理する。
	 */
	private void createGUI() {
		Window titleWindow = new Window(frame);
		titleWindow.setAttributeType(AttributeType.INVERSE);
		titleWindow.setLocation(0, 0);
		titleWindow.setSize(frame.getWidth(), 32);

		Label titleLabel = new Label(new Text("APP_TITLE"));  
		titleLabel.setFont(Font.F16);
		titleLabel.setAttributeType(AttributeType.INVERSE);
		titleLabel.setLocation(0, 0);
		titleLabel.setSize(titleWindow.getWidth(), 16);
		titleWindow.add(titleLabel);
		
		msgLabel = new Label(new Text("MSG_PUSHSTART"));  
		msgLabel.setFont(Font.F16);
		msgLabel.setAttributeType(AttributeType.INVERSE);
		msgLabel.setLocation(0, 16);
		msgLabel.setSize(titleWindow.getWidth(), 16);
		titleWindow.add(msgLabel);

		serialLabel = new Label();
		serialLabel.setSize(titleWindow.getWidth()-200, 16);
		serialLabel.setLocation(
				100 ,
				(frame.getHeight() - titleWindow.getHeight() - msgLabel.getHeight()) / 2);
		frame.add(serialLabel);
		setSerial(null);
		
		frame.addKeyListener(
				new KeyAdapter() {
					public void keyPressed(KeyEvent keyEvent) {
						if (cardManager != null) {
							switch (keyEvent.getKeyCode()) {
							case KeyEvent.KEY_START:     // Start key
								pushStartKey();		
								break;
							case KeyEvent.KEY_CLR_STOP:  // Clear/Stop key
								pushClearStopKey();		
								break;
							}
						}
					}
				}
			);
	}
		
	/**
	 * The process followed after the start key is pressed.
	 * 1.creates and starts a thread which invokes the waitForCard method.
	 * 2.sets LED state.
	 * 
	 * スタートキーが、押された時の処理
	 * 1.waitForCardを実行するスレッドを開始する。
	 * 2.LEDを設定する。
	 */
	synchronized private void pushStartKey() {
		if ((thread==null) || (!thread.isAlive())) {
			setSerial(null);
			
			setMessage(new Text("MSG_SETCARD"));
						
			thread = new Thread(){
				
				/**
				 * 1.invokes the waitForCard method.
				 * 2.if waitForCard interrupted(card==null)
				 *  2-1.displays a message.
				 * 3.if a Card object obtained(card!=null)
				 *  3-1.obtains a CardService object.
				 *  3-2.obtains a mutex lock of the card.
				 *  3-3.obtains serial number.
				 *  3-4.displays serial number.
				 *  3-5.releases the mutex locl of the card.
				 * 4.sets LED state.
				 * 
				 * 1.waitForCardを呼び出す。
				 * 2.中断された場合（card==null）
				 *  2-1.メッセージ表示
				 * 3.カードが、セットされた場合（card!=null)
				 *  3-1.カードサービスを取得する。
				 *  3-2.カードをロックする。
				 *  3-3.シリアル番号を読み込む。
				 *  3-4.シリアル番号を表示する。
				 *  3-5.カードのロックを解除する。
				 * 4.LEDを設定する。
				 */
				public void run() {
					
					Card card = null;
					
					try {
						byte[] serialNo;
						
						card = cardManager.waitForCard(CardManager.INFINITY);
						
						if(card==null){
							setMessage(new Text("ERR_ABORT"));
							
						}else{
							FileAccessService fileAccessService =
								(FileAccessService)card.getCardService(
										FileAccessService.class);
							
							try {
								card.lock();
								
								serialNo = fileAccessService.read(filePath, 0, fileLen);
								
								setSerial(serialNo);
								
								setMessage(new Text("MSG_SUCCESS"));
								
							} finally {
								card.unlock();
							}
						}
						
					} catch (TimeoutException e) {
						e.printStackTrace();
						
					} catch (CardServiceNotFoundException e) {
						setMessage(new Text("ERR_NOCARDSERVICE"));
						e.printStackTrace();
						
					} catch (CardAccessException e) {
						setMessage(new Text("ERR_CARDACCESS"));
						e.printStackTrace();
						
					} catch (VerificationException e) {
						setMessage(new Text("ERR_VERIFY"));
						e.printStackTrace();
						
					} catch (CardServiceException e) {
						setMessage(new Text("ERR_CARDSERVICE"));
						e.printStackTrace();
						
					} finally {
						setLED(Type.START, Operation.GRN_ON);
					}
				}
			};
			thread.start();
			
			setLED(Type.START, Operation.RED_BLINK);
		}
	}
	
	/**
	 * The process followed after the claer/stop key is pressed.
	 * 1.interrupts the thread waiting for cards.
	 * 
	 * クリア/ストップキーが、押された時の処理
	 * 1.abortメソッドを呼び出して、スレッドが停止するのを待つ。
	 */
	synchronized private void pushClearStopKey() {
		if ((thread != null) && thread.isAlive()) {
			
			cardManager.abort();
			
			try {
				thread.join();
			
			} catch (InterruptedException e) {
			}
		}
	}
	
	/**
	 * Displays the given serial number on the serial number label.
	 * 
	 * シリアル番号を、シリアル番号表示ラベルに表示する。
	 */
	private void setSerial(byte[] serialNo) {
		final String replace = "NUMBER";
		Text text = new Text("SERIAL_LABEL");
		
		if (serialNo == null) {
			text.addReplace(replace, "");
		} else { 
			String s="";
			int n = 0;
			for (int i = 0; i < serialNo.length; i++) {
				n = serialNo[i];
				n &=0x000000ff;
				if(n<=0x0000000f){
					s += "0";
				}
				s += Integer.toHexString(n)+" ";
			}
			text.addReplace(replace, s);
		}
		serialLabel.setText(text);
		serialLabel.repaint();
	}
	
	/**
	 * Displays the given message on the message label.
	 * 
	 * メッセージを、メッセージ表示ラベルに表示する。
	 */
	private void setMessage(Text message) {
		msgLabel.setText(message);
		msgLabel.repaint();
	}
	
	/**
	 * Activates LED with the given Type and the given Operation.
	 * 
	 * LEDを動作させる。
	 */
	private void setLED(Type type, Operation ope) {
		try {
			new LED(frame, type, ope).blink();
		} catch (LEDExecutionException e) {
			e.printStackTrace();
		}
	}
}
